package com.ruimo.pluginlib;

import groovy.lang.Binding;
import groovy.lang.GroovyShell;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * <p>PluginRepository class. PluginResitory class reads plugins from
 * the specified directory. All of plugins should be jar files that
 * contains plugin configucation file named 'plugins.conf'. As jar
 * files that do not include the plugin configuration file are simply
 * added to the classpath, you can store helper jar files into the
 * plugin directory as well as the plugin jars.</p>
 *
 * <p>Plugin configuration file is simply executed by PluginRepository
 * as a Groovy script. You can use full functionality of Groovy to
 * initialize plugins.</p>
 *
 * <p>Assume you have created a plugin named MyPlugin.</p>
 
<pre>
public class MyPlugin implements Plugin {
    public void hello() {
        System.out.println("Hello World");
    }

    public String getName() {
        return "Plugin example.";
    }
}
</pre>

 * <p>You need to prepare a plugin configuration file
 * (plugins.conf) for this class:</p>

<pre>
plugins.register(new MyPlugin())
</pre>

 * <p>As this is a Groovy script, you can initialize your plugin as
 * much as you like such as:</p>

<pre>
myPlugin = new MyPlugin()
myPlugin.path = 'contextpath'
myPlugin.timeout = 100
plugins.register(myPlugin)
</pre>

 * <p>If you have more than one plugins, you can store them in a
 * single jar file and register them at the same time such as:</p>

<pre>
plugins.register(new MyPlugin1())
  .register(new MyPlugin2())
  .register(new MyPlugin3())
...
</pre>

 * <p>Or, you can make separate plugin jars for them, of course.</p>
 *
 * <p>Once you finished to create the plugin and it's configuration file,
 * store them into a jar file. The plugin configuration file should be
 * located at the root level of the jar file.</p>

<pre>
myplugin.jar
  |
  +-- plugins.conf
  |
  +-- MyPlugin.class
  |
  ... You can store more than one plugin classes.
  ... You can store non-plugin classes as well.
</pre>

 * <p>To load the plugin in your application, Make an instance of the
 * PluginRepository:</p>

<pre>
File pluginDir = new File("plugins");
PluginRepository&gt;MyPlugin&lt; pluginRepository = new PluginRepository&gt;MyPlugin&lt;(pluginDir);
</pre>

 * <p>Now you can access all of the registed plugins by using {@link
 * #plugins()} method.</p>

<pre>
for (MyPlugin plugin: pluginRepository) {
    plugin.hello();
}
</pre>

 *
 * <p>This class is not thread safe. If you want to use this class in
 * multi-threaded environment. Make sure to synchronize this object by
 * yourself.</p>
 *
 * <p>CAUTION!: Do not add plugin jars into the application
 * classpath. Plugin jars are loaded by a separate class loader. You
 * don't need and must not add plugin jars into your application
 * classpath. Java class is loaded in 'parent first' manner. If you
 * add plugin jars into application classpath, they will be loaded by
 * application class loader (normaly system class loader) but the
 * class loader for plugins. This will cause many hard to debug
 * problems.</p>
 * @author S.Hanai
 */
public class PluginRepository<T extends Plugin> {
    /** Name of the plugin configuration file. */
    public static final String PLUGIN_CONFIG_FILE_NAME = "plugins.conf";

    /**
     * Constructor.
     *
     * Create an instance of {@link #PluginRepository}. The created
     * plugin repository reads jar files in the specified directory.
     * @param pluginDir Directory from where plugin jar files are read.
     * @throws NullPointerException Thrown if pluginDir is null.
     * @throws FileNotFoundException Thrown if pluginDir does not exist.
     * @throws IOException Thrown if I/O error occured.
     */
    public PluginRepository(final File pluginDir) throws IOException {
        if (! pluginDir.exists())
            throw new FileNotFoundException
                ("The plugin directory (" + pluginDir + ") does not exist.");
        if (! pluginDir.isDirectory())
            throw new IOException("This is not a directory (" + pluginDir + ").");

        ClassLoader classLoader = AccessController.doPrivileged
            (new PrivilegedAction<ClassLoader>() {
                public ClassLoader run() {
                    try {
                        return new PluginClassLoader(getClass().getClassLoader(), pluginDir);
                    }
                    catch (IOException ex) {
                        throw new RuntimeException(ex);
                    }
                }
            });

        File[] files = pluginDir.listFiles(new FilenameFilter() {
            public boolean accept(File dir, String name) {
                return name.toLowerCase().endsWith(".jar");
            }
        });
        for (File f:files) {
            JarFile jf = new JarFile(f);
            loadPlugin(classLoader, jf);
        }
    }

    private Set<T> repository = new HashSet<T>();

    /**
     * Return an iteratable object that iterates over all of the
     * regsitered plugins. If no plugins are registered, returned
     * iteratable object ends iteration at once.
     * @return Iteratable object. Null will be never returned.
     */
    public Iterable<T> plugins() {
        return Collections.unmodifiableCollection(new HashSet<T>(repository));
    }

    /**
     * Register a plugin.
     * @param plugin A Plugin object to be registered.
     * @throws NullPointerException Thrown if plugin is null.
     * @throws IllegalArgumentException Thrown if the specified plugin
     * is already registered.
     */
    public PluginRepository<T> register(T plugin) {
        if (plugin == null) throw new NullPointerException();
        if (repository.contains(plugin))
            throw new IllegalArgumentException
                ("The plugin '" + plugin.getName() + "' is already registered.");
        repository.add(plugin);
        return this;
    }
    
    void loadPlugin(ClassLoader classLoader, JarFile jf) throws IOException {
        JarEntry je = jf.getJarEntry(PLUGIN_CONFIG_FILE_NAME);
        if (je == null) return;
        Map<String, Object> map = new HashMap<String, Object>();
        map.put("plugins", this);
        Binding binding = new Binding(map);
        GroovyShell shell = new GroovyShell(classLoader, binding);
        InputStream is = null;
        try {
            is = jf.getInputStream(je);
            shell.evaluate(is, je.getName() + " in " + jf.getName());
        }
        finally {
            if (is != null) {
                try {
                    is.close();
                }
                catch (IOException ex) {
                    // Do not throw this exception. Otherwise, the
                    // exception thrown from try block above (root
                    // cause) will be overwritten.
                    ex.printStackTrace();
                }
            }
        }
    }
}
