package com.ruimo.pluginlib;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

class PluginClassLoader extends ClassLoader {
    final File pluginDirectory;
    final JarFile[] jars;

    PluginClassLoader(ClassLoader parent, File pluginDirectory) throws IOException {
        super (parent);
        if (pluginDirectory == null) throw new NullPointerException();
        this.pluginDirectory = pluginDirectory;
        File[] jarFiles = pluginDirectory.listFiles(new FilenameFilter() {
            public boolean accept(File dir, String name) {
                return name.toLowerCase().endsWith(".jar");
            }
        });
        jars = new JarFile[jarFiles.length];
        for (int i = 0; i < jars.length; ++i) {
            jars[i] = new JarFile(jarFiles[i]);
        }
    }

    protected Class<?> findClass(String name) throws ClassNotFoundException {
        String classFileName = name.replace('.', '/') + ".class";
        for (JarFile jar:jars) {
            JarEntry je = jar.getJarEntry(classFileName);
            if (je != null) {
                byte[] classData = readClassFile(jar, je, classFileName);
                return defineClass(name, classData, 0, classData.length);
            }
        }
        throw new ClassNotFoundException(name);
    }

    byte[] readClassFile(JarFile jar, JarEntry je, String classFileName) {
        byte[] buf = new byte[(int)je.getSize()];
        if (buf.length == 0)
            throw new RuntimeException
                ("Class file is length zero. (" + classFileName + " in " + jar.getName() + ")");
        InputStream is = null;
        try {
            is = jar.getInputStream(je);
            int offset = 0;
            int readSize = 0;
            while (offset < buf.length) {
                readSize = is.read(buf, offset, buf.length - offset);
                if (readSize == -1)
                    throw new RuntimeException
                        ("Unexpected EOF. (" +
                         classFileName + " in " + jar.getName() + ")");
                offset += readSize;
            }
            return buf;
        }
        catch (IOException e) {
            throw new RuntimeException
                ("Class file read error. (" + classFileName + " in " + jar.getName() + ")", e);
        }
        finally {
            if (is != null) {
                try {
                    is.close();
                }
                catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }
}
