package person;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import org.jdesktop.beansbinding.Validator;

public class MinMaxValidator<T extends Number> extends Validator<T> {
    public enum ErrorCode {
        BELOW_MIN, ABOVE_MAX;
    }

    T min;
    T max;

    public void setMin(T min) {
        this.min = min;
    }

    public T getMin() {
        return min;
    }

    public void setMax(T max) {
        this.max = max;
    }

    public T getMax() {
        return max;
    }

    static final Map<Class<? extends Number>, Comparator<? extends Number>>
        comparatorTable = new HashMap<Class<? extends Number>, Comparator<? extends Number>>();
    static {
        comparatorTable.put(Byte.class, new Comparator<Byte>()
            {public int compare(Byte b1, Byte b2) {return b1.compareTo(b2);}});
        comparatorTable.put(Integer.class, new Comparator<Integer>()
            {public int compare(Integer i1, Integer i2) {return i1.compareTo(i2);}});
        comparatorTable.put(Double.class, new Comparator<Double>()
            {public int compare(Double d1, Double d2) {return d1.compareTo(d2);}});
        comparatorTable.put(Float.class, new Comparator<Float>()
            {public int compare(Float f1, Float f2) {return f1.compareTo(f2);}});
        comparatorTable.put(Long.class, new Comparator<Long>()
            {public int compare(Long l1, Long l2) {return l1.compareTo(l2);}});
        comparatorTable.put(Short.class, new Comparator<Short>()
            {public int compare(Short s1, Short s2) {return s1.compareTo(s2);}});
    }

    public final Validator.Result BELOW_MIN
        = new Validator.Result(ErrorCode.BELOW_MIN, "Value is less than minimum value.");
    public final Validator.Result ABOVE_MAX
        = new Validator.Result(ErrorCode.ABOVE_MAX, "Value exceeds maximum value.");

    @Override
    public Validator<T>.Result validate(T value) {
        if (value == null) return null;
        Comparator<T> cmp
            = (Comparator<T>)comparatorTable.get(value.getClass());
        if (cmp == null)
            throw new RuntimeException("Unsupported type:" + value.getClass());
        if (cmp.compare(value, min) < 0) return BELOW_MIN;
        if (cmp.compare(max, value) < 0) return ABOVE_MAX;
        return null;
    }
}
